package com.example.xmppclient.ui

import android.content.Intent
import android.os.Bundle
import android.view.View
import android.widget.Toast
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.example.xmppclient.R
import com.example.xmppclient.databinding.ActivityLoginBinding
import com.example.xmppclient.viewmodel.LoginViewModel
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.launch

@AndroidEntryPoint
class LoginActivity : AppCompatActivity() {

    private lateinit var binding: ActivityLoginBinding
    private val viewModel: LoginViewModel by viewModels()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityLoginBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setupUI()
        observeViewModel()
    }

    private fun setupUI() {
        binding.btnLogin.setOnClickListener {
            val username = binding.etUsername.text.toString().trim()
            val password = binding.etPassword.text.toString().trim()
            val server = binding.etServer.text.toString().trim()
            val port = binding.etPort.text.toString().trim().toIntOrNull() ?: 5222

            if (validateInput(username, password, server)) {
                viewModel.login(username, password, server, port)
            }
        }
    }

    private fun validateInput(username: String, password: String, server: String): Boolean {
        when {
            username.isEmpty() -> {
                binding.etUsername.error = "Username is required"
                return false
            }
            password.isEmpty() -> {
                binding.etPassword.error = "Password is required"
                return false
            }
            server.isEmpty() -> {
                binding.etServer.error = "Server is required"
                return false
            }
            else -> return true
        }
    }

    private fun observeViewModel() {
        lifecycleScope.launch {
            viewModel.loginState.collect { state ->
                when (state) {
                    is LoginViewModel.LoginState.Idle -> {
                        hideLoading()
                    }
                    is LoginViewModel.LoginState.Loading -> {
                        showLoading("Connecting...")
                    }
                    is LoginViewModel.LoginState.Success -> {
                        hideLoading()
                        Toast.makeText(this@LoginActivity, "Connected successfully!", Toast.LENGTH_SHORT).show()
                        startActivity(Intent(this@LoginActivity, MainActivity::class.java))
                        finish()
                    }
                    is LoginViewModel.LoginState.Error -> {
                        hideLoading()
                        showError(state.message)
                    }
                }
            }
        }
    }

    private fun showLoading(message: String) {
        binding.progressBar.visibility = View.VISIBLE
        binding.tvStatus.visibility = View.VISIBLE
        binding.tvStatus.text = message
        binding.btnLogin.isEnabled = false
    }

    private fun hideLoading() {
        binding.progressBar.visibility = View.GONE
        binding.tvStatus.visibility = View.GONE
        binding.btnLogin.isEnabled = true
    }

    private fun showError(message: String) {
        binding.tvStatus.visibility = View.VISIBLE
        binding.tvStatus.text = message
        binding.tvStatus.setTextColor(getColor(R.color.busy_status))
        Toast.makeText(this, message, Toast.LENGTH_LONG).show()
    }
}
