package com.example.xmppclient.ui

import android.content.Intent
import android.os.Bundle
import android.view.Menu
import android.view.MenuItem
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import androidx.viewpager2.adapter.FragmentStateAdapter
import androidx.viewpager2.widget.ViewPager2
import com.example.xmppclient.R
import com.example.xmppclient.databinding.ActivityMainBinding
import com.example.xmppclient.ui.fragments.ContactsFragment
import com.example.xmppclient.ui.fragments.ConversationsFragment
import com.example.xmppclient.viewmodel.MainViewModel
import com.example.xmppclient.xmpp.XMPPConnectionManager
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.launch

@AndroidEntryPoint
class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityMainBinding
    private val viewModel: MainViewModel by viewModels()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)

        setSupportActionBar(binding.toolbar)

        setupViewPager()
        setupBottomNavigation()
        setupFab()
        observeViewModel()
    }

    private fun setupViewPager() {
        val adapter = object : FragmentStateAdapter(this) {
            override fun getItemCount(): Int = 2

            override fun createFragment(position: Int) = when (position) {
                0 -> ConversationsFragment()
                1 -> ContactsFragment()
                else -> ConversationsFragment()
            }
        }

        binding.viewPager.adapter = adapter
        binding.viewPager.registerOnPageChangeCallback(object : ViewPager2.OnPageChangeCallback() {
            override fun onPageSelected(position: Int) {
                binding.bottomNavigation.menu.getItem(position).isChecked = true
                binding.tvTitle.text = when (position) {
                    0 -> getString(R.string.conversations)
                    1 -> getString(R.string.contacts)
                    else -> getString(R.string.conversations)
                }
            }
        })
    }

    private fun setupBottomNavigation() {
        binding.bottomNavigation.setOnItemSelectedListener { item ->
            when (item.itemId) {
                R.id.nav_conversations -> {
                    binding.viewPager.currentItem = 0
                    true
                }
                R.id.nav_contacts -> {
                    binding.viewPager.currentItem = 1
                    true
                }
                else -> false
            }
        }
    }

    private fun setupFab() {
        binding.fabAddContact.setOnClickListener {
            if (binding.viewPager.currentItem == 1) {
                // Show add contact dialog
                viewModel.showAddContactDialog()
            } else {
                // Switch to contacts tab
                binding.viewPager.currentItem = 1
            }
        }
    }

    private fun observeViewModel() {
        lifecycleScope.launch {
            viewModel.connectionState.collect { state ->
                updateConnectionStatus(state)
            }
        }
    }

    private fun updateConnectionStatus(state: XMPPConnectionManager.ConnectionState) {
        val (statusText, statusColor) = when (state) {
            XMPPConnectionManager.ConnectionState.AUTHENTICATED -> {
                getString(R.string.online) to getColor(R.color.online_status)
            }
            XMPPConnectionManager.ConnectionState.CONNECTED -> {
                "Connected" to getColor(R.color.away_status)
            }
            XMPPConnectionManager.ConnectionState.CONNECTING -> {
                "Connecting" to getColor(R.color.away_status)
            }
            XMPPConnectionManager.ConnectionState.DISCONNECTED -> {
                getString(R.string.offline) to getColor(R.color.offline_status)
            }
            XMPPConnectionManager.ConnectionState.ERROR -> {
                "Error" to getColor(R.color.busy_status)
            }
        }

        binding.tvStatus.text = statusText
        binding.statusIndicator.backgroundTintList = android.content.res.ColorStateList.valueOf(statusColor)
    }

    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.main_menu, menu)
        return true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return when (item.itemId) {
            R.id.action_logout -> {
                viewModel.logout()
                startActivity(Intent(this, LoginActivity::class.java))
                finish()
                true
            }
            R.id.action_settings -> {
                // TODO: Open settings
                true
            }
            else -> super.onOptionsItemSelected(item)
        }
    }
}
