package com.example.xmppclient.ui.adapter

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import com.example.xmppclient.R
import com.example.xmppclient.databinding.ItemContactBinding
import com.example.xmppclient.viewmodel.ContactsViewModel

class ContactAdapter(
    private val onItemClick: (ContactsViewModel.ContactItem) -> Unit
) : ListAdapter<ContactsViewModel.ContactItem, ContactAdapter.ContactViewHolder>(ContactDiffCallback()) {

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ContactViewHolder {
        val binding = ItemContactBinding.inflate(
            LayoutInflater.from(parent.context),
            parent,
            false
        )
        return ContactViewHolder(binding)
    }

    override fun onBindViewHolder(holder: ContactViewHolder, position: Int) {
        holder.bind(getItem(position))
    }

    inner class ContactViewHolder(
        private val binding: ItemContactBinding
    ) : RecyclerView.ViewHolder(binding.root) {

        fun bind(contact: ContactsViewModel.ContactItem) {
            binding.apply {
                tvAvatar.text = contact.avatar
                tvContactName.text = contact.name
                tvContactJid.text = contact.jid

                // Show/hide status message
                if (contact.statusMessage.isNullOrBlank()) {
                    tvStatusMessage.visibility = View.GONE
                } else {
                    tvStatusMessage.visibility = View.VISIBLE
                    tvStatusMessage.text = contact.statusMessage
                }

                // Set presence status
                val (presenceColor, presenceText) = when (contact.presenceStatus) {
                    "online" -> R.color.online_status to "Online"
                    "away" -> R.color.away_status to "Away"
                    "busy" -> R.color.busy_status to "Busy"
                    else -> R.color.offline_status to "Offline"
                }

                presenceIndicator.backgroundTintList =
                    android.content.res.ColorStateList.valueOf(root.context.getColor(presenceColor))
                tvPresenceStatus.text = presenceText

                root.setOnClickListener {
                    onItemClick(contact)
                }
            }
        }
    }

    private class ContactDiffCallback : DiffUtil.ItemCallback<ContactsViewModel.ContactItem>() {
        override fun areItemsTheSame(
            oldItem: ContactsViewModel.ContactItem,
            newItem: ContactsViewModel.ContactItem
        ): Boolean {
            return oldItem.jid == newItem.jid
        }

        override fun areContentsTheSame(
            oldItem: ContactsViewModel.ContactItem,
            newItem: ContactsViewModel.ContactItem
        ): Boolean {
            return oldItem == newItem
        }
    }
}
