package com.example.xmppclient.ui.adapter

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import androidx.recyclerview.widget.RecyclerView
import com.example.xmppclient.R
import com.example.xmppclient.databinding.ItemConversationBinding
import com.example.xmppclient.viewmodel.ConversationsViewModel
import java.text.SimpleDateFormat
import java.util.*

class ConversationAdapter(
    private val onItemClick: (ConversationsViewModel.ConversationItem) -> Unit
) : ListAdapter<ConversationsViewModel.ConversationItem, ConversationAdapter.ConversationViewHolder>(ConversationDiffCallback()) {

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ConversationViewHolder {
        val binding = ItemConversationBinding.inflate(
            LayoutInflater.from(parent.context),
            parent,
            false
        )
        return ConversationViewHolder(binding)
    }

    override fun onBindViewHolder(holder: ConversationViewHolder, position: Int) {
        holder.bind(getItem(position))
    }

    inner class ConversationViewHolder(
        private val binding: ItemConversationBinding
    ) : RecyclerView.ViewHolder(binding.root) {

        fun bind(conversation: ConversationsViewModel.ConversationItem) {
            binding.apply {
                tvAvatar.text = conversation.avatar
                tvContactName.text = conversation.contactName
                tvLastMessage.text = conversation.lastMessage
                tvTimestamp.text = formatTimestamp(conversation.timestamp)

                // Show/hide unread indicator
                unreadIndicator.visibility = if (conversation.unreadCount > 0) View.VISIBLE else View.GONE

                // Set presence indicator color
                val presenceColor = when (conversation.presenceStatus) {
                    "online" -> R.color.online_status
                    "away" -> R.color.away_status
                    "busy" -> R.color.busy_status
                    else -> R.color.offline_status
                }
                presenceIndicator.backgroundTintList =
                    android.content.res.ColorStateList.valueOf(root.context.getColor(presenceColor))

                root.setOnClickListener {
                    onItemClick(conversation)
                }
            }
        }

        private fun formatTimestamp(timestamp: Long): String {
            val now = System.currentTimeMillis()
            val diff = now - timestamp

            return when {
                diff < 60 * 1000 -> "Now" // Less than 1 minute
                diff < 60 * 60 * 1000 -> "${diff / (60 * 1000)}m" // Less than 1 hour
                diff < 24 * 60 * 60 * 1000 -> "${diff / (60 * 60 * 1000)}h" // Less than 1 day
                diff < 7 * 24 * 60 * 60 * 1000 -> {
                    SimpleDateFormat("EEE", Locale.getDefault()).format(Date(timestamp))
                } // Less than 1 week
                else -> {
                    SimpleDateFormat("MMM dd", Locale.getDefault()).format(Date(timestamp))
                } // More than 1 week
            }
        }
    }

    private class ConversationDiffCallback : DiffUtil.ItemCallback<ConversationsViewModel.ConversationItem>() {
        override fun areItemsTheSame(
            oldItem: ConversationsViewModel.ConversationItem,
            newItem: ConversationsViewModel.ConversationItem
        ): Boolean {
            return oldItem.contactJid == newItem.contactJid
        }

        override fun areContentsTheSame(
            oldItem: ConversationsViewModel.ConversationItem,
            newItem: ConversationsViewModel.ConversationItem
        ): Boolean {
            return oldItem == newItem
        }
    }
}
