package com.example.xmppclient.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.example.xmppclient.data.entity.Message
import com.example.xmppclient.repository.MessageRepository
import com.example.xmppclient.xmpp.XMPPConnectionManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class ChatViewModel @Inject constructor(
    private val messageRepository: MessageRepository,
    private val xmppManager: XMPPConnectionManager
) : ViewModel() {

    private val _messages = MutableStateFlow<List<Message>>(emptyList())
    val messages: StateFlow<List<Message>> = _messages

    fun loadMessages(contactJid: String) {
        viewModelScope.launch {
            val userJid = xmppManager.getCurrentUserJid() ?: return@launch
            messageRepository.getConversation(userJid, contactJid).collect { messageList ->
                _messages.value = messageList
            }
        }
    }

    fun sendMessage(toJid: String, messageBody: String) {
        viewModelScope.launch {
            val success = xmppManager.sendMessage(toJid, messageBody)
            if (success) {
                val message = Message(
                    fromJid = xmppManager.getCurrentUserJid() ?: "",
                    toJid = toJid,
                    body = messageBody,
                    timestamp = System.currentTimeMillis(),
                    isOutgoing = true,
                    isDelivered = true
                )
                messageRepository.insertMessage(message)
            }
        }
    }
}
