package com.example.xmppclient.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.example.xmppclient.data.entity.Contact
import com.example.xmppclient.repository.ContactRepository
import com.example.xmppclient.xmpp.XMPPConnectionManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class ContactsViewModel @Inject constructor(
    private val contactRepository: ContactRepository,
    private val xmppManager: XMPPConnectionManager
) : ViewModel() {

    private val _contacts = MutableStateFlow<List<ContactItem>>(emptyList())
    val contacts: StateFlow<List<ContactItem>> = _contacts

    private val _addContactResult = MutableStateFlow<AddContactResult?>(null)
    val addContactResult: StateFlow<AddContactResult?> = _addContactResult

    data class ContactItem(
        val jid: String,
        val name: String,
        val presenceStatus: String,
        val statusMessage: String?,
        val avatar: String,
        val subscription: String
    )

    sealed class AddContactResult {
        object Success : AddContactResult()
        data class Error(val message: String) : AddContactResult()
    }

    init {
        loadContacts()
        syncRosterContacts()
    }

    private fun loadContacts() {
        viewModelScope.launch {
            contactRepository.getAllContacts()
                .map { contacts ->
                    contacts.map { contact ->
                        ContactItem(
                            jid = contact.jid,
                            name = contact.name,
                            presenceStatus = contact.presenceStatus,
                            statusMessage = contact.statusMessage,
                            avatar = contact.name.first().uppercase(),
                            subscription = contact.subscription
                        )
                    }
                }
                .collect { contactItems ->
                    _contacts.value = contactItems
                }
        }
    }

    private fun syncRosterContacts() {
        viewModelScope.launch {
            try {
                val rosterEntries = xmppManager.getRosterEntries()
                val contacts = rosterEntries.map { entry ->
                    Contact(
                        jid = entry.jid.asBareJid().toString(),
                        name = entry.name ?: entry.jid.localpart.toString(),
                        subscription = entry.type.toString(),
                        presenceStatus = "offline" // Will be updated by presence listener
                    )
                }
                contactRepository.insertContacts(contacts)
            } catch (e: Exception) {
                // Handle error
            }
        }
    }

    fun addContact(jid: String, name: String) {
        viewModelScope.launch {
            try {
                // Add to XMPP roster
                val success = xmppManager.addContact(jid, name)

                if (success) {
                    // Add to local database
                    val contact = Contact(
                        jid = jid,
                        name = name,
                        subscription = "none",
                        presenceStatus = "offline"
                    )
                    contactRepository.insertContact(contact)
                    _addContactResult.value = AddContactResult.Success
                } else {
                    _addContactResult.value = AddContactResult.Error("Failed to add contact")
                }
            } catch (e: Exception) {
                _addContactResult.value = AddContactResult.Error(e.message ?: "Unknown error")
            }
        }
    }

    fun removeContact(jid: String) {
        viewModelScope.launch {
            try {
                val success = xmppManager.removeContact(jid)
                if (success) {
                    val contact = contactRepository.getContact(jid)
                    contact?.let { contactRepository.deleteContact(it) }
                }
            } catch (e: Exception) {
                // Handle error
            }
        }
    }

    fun clearAddContactResult() {
        _addContactResult.value = null
    }
}
