package com.example.xmppclient.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.example.xmppclient.data.entity.Contact
import com.example.xmppclient.data.entity.Message
import com.example.xmppclient.repository.ContactRepository
import com.example.xmppclient.repository.MessageRepository
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class ConversationsViewModel @Inject constructor(
    private val messageRepository: MessageRepository,
    private val contactRepository: ContactRepository
) : ViewModel() {

    private val _conversations = MutableStateFlow<List<ConversationItem>>(emptyList())
    val conversations: StateFlow<List<ConversationItem>> = _conversations

    data class ConversationItem(
        val contactJid: String,
        val contactName: String,
        val lastMessage: String,
        val timestamp: Long,
        val unreadCount: Int,
        val presenceStatus: String,
        val avatar: String
    )

    init {
        loadConversations()
    }

    private fun loadConversations() {
        viewModelScope.launch {
            combine(
                messageRepository.getAllMessages(),
                contactRepository.getAllContacts()
            ) { messages, contacts ->
                buildConversationList(messages, contacts)
            }.collect { conversationList ->
                _conversations.value = conversationList
            }
        }
    }

    private fun buildConversationList(messages: List<Message>, contacts: List<Contact>): List<ConversationItem> {
        // Group messages by contact JID
        val messageGroups = messages.groupBy { message ->
            if (message.isOutgoing) message.toJid else message.fromJid
        }

        // Create conversations from message groups
        val conversations = mutableListOf<ConversationItem>()

        messageGroups.forEach { (contactJid, messageList) ->
            val contact = contacts.find { it.jid == contactJid }
            val lastMessage = messageList.maxByOrNull { it.timestamp }
            val unreadCount = messageList.count { !it.isRead && !it.isOutgoing }

            if (lastMessage != null) {
                conversations.add(
                    ConversationItem(
                        contactJid = contactJid,
                        contactName = contact?.name ?: contactJid.substringBefore("@"),
                        lastMessage = lastMessage.body,
                        timestamp = lastMessage.timestamp,
                        unreadCount = unreadCount,
                        presenceStatus = contact?.presenceStatus ?: "offline",
                        avatar = (contact?.name ?: contactJid).first().uppercase()
                    )
                )
            }
        }

        // Sort by timestamp (newest first)
        return conversations.sortedByDescending { it.timestamp }
    }
}
