package com.example.xmppclient.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.example.xmppclient.repository.ContactRepository
import com.example.xmppclient.repository.MessageRepository
import com.example.xmppclient.xmpp.XMPPConnectionManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class MainViewModel @Inject constructor(
    private val xmppManager: XMPPConnectionManager,
    private val contactRepository: ContactRepository,
    private val messageRepository: MessageRepository
) : ViewModel() {

    val connectionState: StateFlow<XMPPConnectionManager.ConnectionState> = xmppManager.connectionState

    private val _showAddContactDialog = MutableStateFlow(false)
    val showAddContactDialog: StateFlow<Boolean> = _showAddContactDialog

    init {
        observeXMPPEvents()
    }

    private fun observeXMPPEvents() {
        viewModelScope.launch {
            // Observe incoming messages
            xmppManager.messageReceived.collect { receivedMessage ->
                receivedMessage?.let { message ->
                    val messageEntity = com.example.xmppclient.data.entity.Message(
                        fromJid = message.from,
                        toJid = xmppManager.getCurrentUserJid() ?: "",
                        body = message.body,
                        timestamp = message.timestamp,
                        isOutgoing = false,
                        isDelivered = true
                    )
                    messageRepository.insertMessage(messageEntity)
                }
            }
        }

        viewModelScope.launch {
            // Observe presence changes
            xmppManager.presenceChanged.collect { presenceUpdate ->
                presenceUpdate?.let { update ->
                    contactRepository.updatePresence(
                        jid = update.jid,
                        status = update.status,
                        statusMessage = update.statusMessage,
                        lastSeen = if (update.status == "offline") System.currentTimeMillis() else null
                    )
                }
            }
        }
    }

    fun showAddContactDialog() {
        _showAddContactDialog.value = true
    }

    fun hideAddContactDialog() {
        _showAddContactDialog.value = false
    }

    fun logout() {
        xmppManager.disconnect()
    }
}
