package com.example.xmppclient.xmpp

import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import org.jivesoftware.smack.ConnectionConfiguration
import org.jivesoftware.smack.ReconnectionManager
import org.jivesoftware.smack.chat2.Chat
import org.jivesoftware.smack.chat2.ChatManager
import org.jivesoftware.smack.packet.Message
import org.jivesoftware.smack.packet.Presence
import org.jivesoftware.smack.roster.Roster
import org.jivesoftware.smack.tcp.XMPPTCPConnection
import org.jivesoftware.smack.tcp.XMPPTCPConnectionConfiguration
import org.jxmpp.jid.EntityBareJid
import org.jxmpp.jid.impl.JidCreate
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
class XMPPConnectionManager @Inject constructor() {

    private val tag = "XMPPConnectionManager"

    private var connection: XMPPTCPConnection? = null
    private var chatManager: ChatManager? = null
    private var roster: Roster? = null

    private val _connectionState = MutableStateFlow(ConnectionState.DISCONNECTED)
    val connectionState: StateFlow<ConnectionState> = _connectionState

    private val _messageReceived = MutableStateFlow<ReceivedMessage?>(null)
    val messageReceived: StateFlow<ReceivedMessage?> = _messageReceived

    private val _presenceChanged = MutableStateFlow<PresenceUpdate?>(null)
    val presenceChanged: StateFlow<PresenceUpdate?> = _presenceChanged

    private val scope = CoroutineScope(Dispatchers.IO)

    enum class ConnectionState {
        DISCONNECTED, CONNECTING, CONNECTED, AUTHENTICATED, ERROR
    }

    data class ReceivedMessage(
        val from: String,
        val body: String,
        val timestamp: Long
    )

    data class PresenceUpdate(
        val jid: String,
        val status: String,
        val statusMessage: String?
    )

    suspend fun connect(username: String, password: String, server: String, port: Int = 5222): Boolean {
        return try {
            _connectionState.value = ConnectionState.CONNECTING
            Log.d(tag, "Attempting to connect to $server:$port with user $username")

            val config = XMPPTCPConnectionConfiguration.builder()
                .setUsernameAndPassword(username, password)
                .setXmppDomain(server)
                .setHost(server)
                .setPort(port)
                .setSecurityMode(ConnectionConfiguration.SecurityMode.ifpossible)
                .build()

            connection = XMPPTCPConnection(config)

            connection?.let { conn ->
                conn.connect()
                _connectionState.value = ConnectionState.CONNECTED
                Log.d(tag, "Connected to server")

                conn.login()
                _connectionState.value = ConnectionState.AUTHENTICATED
                Log.d(tag, "Authenticated successfully")

                setupManagers()
                setupListeners()

                // Enable automatic reconnection
                ReconnectionManager.getInstanceFor(conn).enableAutomaticReconnection()

                true
            } ?: false

        } catch (e: Exception) {
            Log.e(tag, "Connection failed", e)
            _connectionState.value = ConnectionState.ERROR
            false
        }
    }

    private fun setupManagers() {
        connection?.let { conn ->
            chatManager = ChatManager.getInstanceFor(conn)
            roster = Roster.getInstanceFor(conn)

            // Load roster
            roster?.reloadAndWait()
        }
    }

    private fun setupListeners() {
        connection?.let { conn ->
            // Message listener
            chatManager?.addIncomingListener { from, message, chat ->
                scope.launch {
                    Log.d(tag, "Received message from ${from.asBareJid()}: ${message.body}")
                    _messageReceived.value = ReceivedMessage(
                        from = from.asBareJid().toString(),
                        body = message.body ?: "",
                        timestamp = System.currentTimeMillis()
                    )
                }
            }

            // Presence listener
            roster?.addPresenceEventListener { presence ->
                scope.launch {
                    Log.d(tag, "Presence update: ${presence.from} - ${presence.type}")
                    val status = when (presence.type) {
                        Presence.Type.available -> when (presence.mode) {
                            Presence.Mode.away -> "away"
                            Presence.Mode.dnd -> "busy"
                            else -> "online"
                        }
                        Presence.Type.unavailable -> "offline"
                        else -> "offline"
                    }

                    _presenceChanged.value = PresenceUpdate(
                        jid = presence.from.asBareJid().toString(),
                        status = status,
                        statusMessage = presence.status
                    )
                }
            }

            // Connection listener
            conn.addConnectionListener(object : org.jivesoftware.smack.ConnectionListener {
                override fun connected(connection: org.jivesoftware.smack.XMPPConnection) {
                    Log.d(tag, "Connection established")
                    _connectionState.value = ConnectionState.CONNECTED
                }

                override fun authenticated(connection: org.jivesoftware.smack.XMPPConnection, resumed: Boolean) {
                    Log.d(tag, "Authentication successful")
                    _connectionState.value = ConnectionState.AUTHENTICATED
                }

                override fun connectionClosed() {
                    Log.d(tag, "Connection closed")
                    _connectionState.value = ConnectionState.DISCONNECTED
                }

                override fun connectionClosedOnError(e: Exception) {
                    Log.e(tag, "Connection closed on error", e)
                    _connectionState.value = ConnectionState.ERROR
                }

                override fun reconnectionSuccessful() {
                    Log.d(tag, "Reconnection successful")
                    _connectionState.value = ConnectionState.AUTHENTICATED
                }

                override fun reconnectingIn(seconds: Int) {
                    Log.d(tag, "Reconnecting in $seconds seconds")
                }

                override fun reconnectionFailed(e: Exception) {
                    Log.e(tag, "Reconnection failed", e)
                    _connectionState.value = ConnectionState.ERROR
                }
            })
        }
    }

    suspend fun sendMessage(toJid: String, messageBody: String): Boolean {
        return try {
            val bareJid = JidCreate.entityBareFrom(toJid)
            val chat = chatManager?.chatWith(bareJid)

            val message = org.jivesoftware.smack.packet.Message()
            message.body = messageBody
            message.type = Message.Type.chat

            chat?.send(message)
            Log.d(tag, "Message sent to $toJid: $messageBody")
            true
        } catch (e: Exception) {
            Log.e(tag, "Failed to send message to $toJid", e)
            false
        }
    }

    suspend fun addContact(jid: String, name: String): Boolean {
        return try {
            val bareJid = JidCreate.entityBareFrom(jid)
            roster?.createEntry(bareJid, name, null)
            Log.d(tag, "Contact added: $jid")
            true
        } catch (e: Exception) {
            Log.e(tag, "Failed to add contact: $jid", e)
            false
        }
    }

    suspend fun removeContact(jid: String): Boolean {
        return try {
            val bareJid = JidCreate.entityBareFrom(jid)
            val entry = roster?.getEntry(bareJid)
            entry?.let { roster?.removeEntry(it) }
            Log.d(tag, "Contact removed: $jid")
            true
        } catch (e: Exception) {
            Log.e(tag, "Failed to remove contact: $jid", e)
            false
        }
    }

    fun setPresence(status: String, statusMessage: String? = null) {
        try {
            val presence = Presence(Presence.Type.available)
            when (status) {
                "away" -> presence.mode = Presence.Mode.away
                "busy" -> presence.mode = Presence.Mode.dnd
                "online" -> presence.mode = Presence.Mode.available
            }
            statusMessage?.let { presence.status = it }

            connection?.sendStanza(presence)
            Log.d(tag, "Presence updated: $status")
        } catch (e: Exception) {
            Log.e(tag, "Failed to set presence", e)
        }
    }

    fun getRosterEntries(): List<org.jivesoftware.smack.roster.RosterEntry> {
        return roster?.entries?.toList() ?: emptyList()
    }

    fun getCurrentUserJid(): String? {
        return connection?.user?.asBareJid()?.toString()
    }

    fun disconnect() {
        try {
            connection?.disconnect()
            _connectionState.value = ConnectionState.DISCONNECTED
            Log.d(tag, "Disconnected from server")
        } catch (e: Exception) {
            Log.e(tag, "Error during disconnect", e)
        }
    }

    fun isConnected(): Boolean {
        return connection?.isConnected == true && connection?.isAuthenticated == true
    }
}
