# server.py
import asyncio
import uuid
from dataclasses import dataclass, field
from typing import Optional, Dict

from fastapi import FastAPI, WebSocket, WebSocketDisconnect, HTTPException, Response
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse, FileResponse, PlainTextResponse
from fastapi.staticfiles import StaticFiles
from pydantic import BaseModel

import httpx, re

app = FastAPI(title="KISS-PGPfone rendezvous/relay + web")

# CORS: permissive for dev; tighten if serving same-origin only.
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# --- Serve the web client ---
app.mount("/web", StaticFiles(directory="web", html=True), name="web")

@app.get("/", include_in_schema=False)
async def index_root():
    return FileResponse("web/index.html")

# --- Self-host Noble modules via proxy to avoid CSP/CORS/MIME issues ---
@app.get("/vendor/{path:path}")
async def vendor(path: str):
    base = "https://esm.sh/"
    url = base + path
    async with httpx.AsyncClient(follow_redirects=True, timeout=30) as client:
        r = await client.get(url, headers={"User-Agent": "Mozilla/5.0"})
    text = r.text
    # Rewrite any absolute esm.sh imports to our local /vendor
    text = re.sub(r'(from|import|export\s+\*\s*from)\s+["\']https://esm\\.sh/', r'\1 "/vendor/', text)
    text = re.sub(r'import\s+"https://esm\\.sh/', 'import "/vendor/', text)
    text = re.sub(r'from\s+"/(v\\d+[^"]*)"', r'from "/vendor/\\1"', text)
    # Generic: any absolute import path not already under /vendor -> /vendor/
    text = re.sub(r'(from|import|export\s+\*\s*from)\s+["\']/(?!vendor/)([^"\']+)["\']', r'\1 "/vendor/\2"', text)  # absroot_rewrite

    text = re.sub(r'import\s+"/(v\\d+[^"]*)"', r'import "/vendor/\\1"', text)
    text = re.sub(r'import\s+["\']/(?!vendor/)([^"\']+)["\']', r'import "/vendor/\1"', text)
    ct = r.headers.get("content-type", "")
    if "javascript" not in ct:
        ct = "application/javascript; charset=utf-8"
    return Response(text, media_type=ct)

# Aliases so absolute imports like "/@noble/..." or "/v135/..." resolve on our origin.
@app.get("/@noble/{path:path}")
async def vendor_noble(path: str):
    return await vendor(f"@noble/{path}")

@app.get("/es2022/{path:path}")
async def vendor_es(path: str):
    return await vendor(f"es2022/{path}")

@app.get("/v{rest:path}")
async def vendor_v(rest: str):
    # rest begins with digits and a slash as esm.sh version prefix; pass through
    return await vendor(f"v{rest}")


# --- Rendezvous/relay API ---
@dataclass
class Party:
    pubkey_b64: Optional[str] = None
    ws: Optional[WebSocket] = None

@dataclass
class Session:
    id: str
    caller: Party = field(default_factory=Party)
    callee: Party = field(default_factory=Party)

SESSIONS: Dict[str, Session] = {}

class PubKey(BaseModel):
    pubkey: str  # base64

@app.get("/healthz", include_in_schema=False)
async def healthz():
    return PlainTextResponse("ok")

@app.post("/session")
def create_session():
    sid = uuid.uuid4().hex
    SESSIONS[sid] = Session(id=sid)
    return {"session_id": sid}

@app.post("/session/{sid}/offer")
def post_offer(sid: str, pk: PubKey):
    s = SESSIONS.get(sid)
    if not s:
        raise HTTPException(404, "unknown session")
    s.caller.pubkey_b64 = pk.pubkey
    return {"status": "ok"}

@app.post("/session/{sid}/answer")
def post_answer(sid: str, pk: PubKey):
    s = SESSIONS.get(sid)
    if not s:
        raise HTTPException(404, "unknown session")
    s.callee.pubkey_b64 = pk.pubkey
    return {"status": "ok"}

@app.get("/session/{sid}/peerkey")
def get_peerkey(sid: str, role: str):
    s = SESSIONS.get(sid)
    if not s:
        raise HTTPException(404, "unknown session")
    if role == "caller":
        return JSONResponse({"peer_pubkey": s.callee.pubkey_b64})
    if role == "callee":
        return JSONResponse({"peer_pubkey": s.caller.pubkey_b64})
    raise HTTPException(400, "role must be 'caller' or 'callee'")

@app.websocket("/ws/{sid}")
async def media_ws(ws: WebSocket, sid: str, role: str):
    s = SESSIONS.get(sid)
    if not s:
        await ws.close(code=4404)
        return

    await ws.accept()
    party = s.caller if role == "caller" else s.callee if role == "callee" else None
    other = s.callee if role == "caller" else s.caller if role == "callee" else None
    if party is None:
        await ws.close(code=4400)
        return

    party.ws = ws

    # Wait until both are connected
    while other.ws is None:
        await asyncio.sleep(0.05)

    try:
        recv_task = asyncio.create_task(pipe(ws, other.ws))
        send_task = asyncio.create_task(pipe(other.ws, ws))
        done, pending = await asyncio.wait({recv_task, send_task}, return_when=asyncio.FIRST_EXCEPTION)
        for t in pending:
            t.cancel()
    except WebSocketDisconnect:
        pass
    finally:
        try:
            await ws.close()
        except Exception:
            pass
        party.ws = None
        if (s.caller.ws is None) and (s.callee.ws is None):
            SESSIONS.pop(sid, None)

async def pipe(src: WebSocket, dst: WebSocket):
    while True:
        data = await src.receive_bytes()
        await dst.send_bytes(data)

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)
