
# client.py
import argparse, asyncio, base64, hmac, struct, sys
import httpx, numpy as np, sounddevice as sd, websockets
from cryptography.hazmat.primitives.asymmetric.x25519 import X25519PrivateKey, X25519PublicKey
from cryptography.hazmat.primitives.kdf.hkdf import HKDF
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.ciphers.aead import ChaCha20Poly1305

API = "http://127.0.0.1:8000"
SAMPLE_RATE = 16000; SAMPLES_PER_FRAME = 320; BYTES_PER_FRAME = 640
b64e = lambda b: base64.b64encode(b).decode(); b64d = lambda s: base64.b64decode(s)

def derive_keys(my_pub, peer_pub, shared):
    lo, hi = sorted([my_pub, peer_pub]); info = b"KISS-PGPfone-v1|" + lo + b"|" + hi
    material = HKDF(algorithm=hashes.SHA256(), length=80, salt=None, info=info).derive(shared)
    k0, k1 = material[:32], material[32:64]; p0, p1 = material[64:68], material[68:72]
    return (k0,k1,p0,p1) if my_pub==lo else (k1,k0,p1,p0)

def sas(shared, my_pub, peer_pub):
    lo, hi = sorted([my_pub, peer_pub]); mac = hmac.new(shared, b"KISS-PGPfone-SAS|" + lo + b"|" + hi, "sha256").digest()
    return base64.b32encode(mac)[:10].decode().rstrip("=")

class E2EE:
    def __init__(self, sk, rk, sp, rp): self.tx=ChaCha20Poly1305(sk); self.rx=ChaCha20Poly1305(rk); self.sp=sp; self.rp=rp; self.c=0
    def seal(self, pt): import struct; n=self.sp+struct.pack(">Q",self.c); self.c+=1; return struct.pack(">Q",self.c-1)+self.tx.encrypt(n,pt,None)
    def open(self, fr): import struct; (ctr,)=struct.unpack(">Q",fr[:8]); n=self.rp+struct.pack(">Q",ctr); return self.rx.decrypt(n,fr[8:],None)

def rec(q):
    def cb(indata, frames, time, status): q.put_nowait(bytes(indata))
    s = sd.InputStream(channels=1, samplerate=SAMPLE_RATE, dtype="int16", blocksize=SAMPLES_PER_FRAME, callback=cb); s.start(); return s
def play(q):
    def cb(outdata, frames, time, status):
        try: chunk = q.get_nowait()
        except asyncio.QueueEmpty: chunk = b"\x00"*BYTES_PER_FRAME
        outdata[:] = np.frombuffer(chunk, dtype=np.int16).reshape(-1,1)
    s = sd.OutputStream(channels=1, samplerate=SAMPLE_RATE, dtype="int16", blocksize=SAMPLES_PER_FRAME, callback=cb); s.start(); return s

async def run(role, api, sid):
    async with httpx.AsyncClient(timeout=10) as http:
        sk = X25519PrivateKey.generate()
        pk = sk.public_key().public_bytes(encoding=serialization.Encoding.Raw, format=serialization.PublicFormat.Raw)
        if role=="caller":
            if not sid: sid=(await http.post(f"{api}/session")).json()["session_id"]; print("Session:", sid)
            await http.post(f"{api}/session/{sid}/offer", json={"pubkey": b64e(pk)})
            peer=None; print("Waiting for callee...")
            while not peer: peer=(await http.get(f"{api}/session/{sid}/peerkey", params={"role":"caller"})).json().get("peer_pubkey"); await asyncio.sleep(0.5)
        else:
            if not sid: print("--session-id required", file=sys.stderr); sys.exit(2)
            await http.post(f"{api}/session/{sid}/answer", json={"pubkey": b64e(pk)})
            peer=None
            while not peer: peer=(await http.get(f"{api}/session/{sid}/peerkey", params={"role":"callee"})).json().get("peer_pubkey"); await asyncio.sleep(0.5)
        peer_pk = b64d(peer); shared = sk.exchange(X25519PublicKey.from_public_bytes(peer_pk))
        skey,rkey,sp,rp = derive_keys(pk, peer_pk, shared)
        print("SAS:", sas(shared, pk, peer_pk)); input("If matched, press Enter... ")
        e2ee = E2EE(skey,rkey,sp,rp)
        uri = f"{api.replace('http','ws')}/ws/{sid}?role={role}"
        async with websockets.connect(uri, max_size=None) as ws:
            iq, oq = asyncio.Queue(maxsize=50), asyncio.Queue(maxsize=50)
            rec(iq); play(oq)
            async def sender(): 
                while True: await ws.send(e2ee.seal(await iq.get()))
            async def receiver():
                while True:
                    m = await ws.recv()
                    if isinstance(m,(bytes,bytearray)):
                        try: await oq.put(e2ee.open(m))
                        except Exception: pass
            await asyncio.gather(sender(), receiver())

if __name__ == "__main__":
    p = argparse.ArgumentParser(); p.add_argument("role", choices=["caller","callee"]); p.add_argument("--api", default=API); p.add_argument("--session-id"); a=p.parse_args()
    try: asyncio.run(run(a.role, a.api, a.session_id))
    except KeyboardInterrupt: pass
