import React, { useMemo, useState } from "react";

// CSAT = (Good / (Good + Bad)) * 100
// Nice, accessible, and responsive UI with TailwindCSS.

export default function CSATCalculator() {
  const [good, setGood] = useState<number | "">(43);
  const [bad, setBad] = useState<number | "">(3);

  const parsedGood = typeof good === "string" ? Number.NaN : good;
  const parsedBad = typeof bad === "string" ? Number.NaN : bad;

  const { total, ratio, percent } = useMemo(() => {
    const g = Number.isFinite(parsedGood) ? parsedGood : 0;
    const b = Number.isFinite(parsedBad) ? parsedBad : 0;
    const t = g + b;
    const r = t > 0 ? g / t : 0;
    const p = r * 100;
    return { total: t, ratio: r, percent: p };
  }, [parsedGood, parsedBad]);

  const fmt = (n: number, digits = 2) =>
    Number.isFinite(n) ? n.toLocaleString(undefined, { maximumFractionDigits: digits, minimumFractionDigits: digits }) : "—";

  const onNumberChange = (
    e: React.ChangeEvent<HTMLInputElement>,
    setter: React.Dispatch<React.SetStateAction<number | "">>
  ) => {
    const v = e.target.value.trim();
    if (v === "") {
      setter("");
      return;
    }
    // Only allow non-negative integers
    const int = Number(v);
    if (/^\d+$/.test(v) && Number.isFinite(int) && int >= 0) {
      setter(int);
    }
  };

  const reset = () => {
    setGood(0);
    setBad(0);
  };

  const loadExample = () => {
    setGood(43);
    setBad(3); // total 46
  };

  const invalid = (good === "" || bad === "") || !Number.isFinite(parsedGood) || !Number.isFinite(parsedBad);

  return (
    <div className="min-h-screen w-full bg-gradient-to-br from-slate-900 via-slate-800 to-indigo-900 text-slate-100 flex items-center justify-center p-4">
      <div className="w-full max-w-3xl">
        <header className="mb-6 text-center">
          <h1 className="text-3xl md:text-4xl font-semibold tracking-tight">CSAT Calculator</h1>
          <p className="text-slate-300 mt-2">Customer Satisfaction (CSAT) = <span className="font-mono">(Good ÷ (Good + Bad)) × 100</span></p>
        </header>

        <div className="grid md:grid-cols-2 gap-4">
          <section className="rounded-2xl bg-slate-950/40 backdrop-blur border border-white/10 p-5 shadow-xl shadow-black/30">
            <h2 className="text-lg font-medium mb-4">Inputs</h2>
            <div className="space-y-4">
              <div>
                <label htmlFor="good" className="block text-sm text-slate-300 mb-1">Good ratings</label>
                <input
                  id="good"
                  inputMode="numeric"
                  pattern="[0-9]*"
                  value={good}
                  onChange={(e) => onNumberChange(e, setGood)}
                  placeholder="e.g., 43"
                  className="w-full rounded-xl bg-slate-900 border border-white/10 px-4 py-3 outline-none focus:ring-2 focus:ring-indigo-400/60 focus:border-indigo-400/60 font-medium"
                />
              </div>
              <div>
                <label htmlFor="bad" className="block text-sm text-slate-300 mb-1">Bad ratings</label>
                <input
                  id="bad"
                  inputMode="numeric"
                  pattern="[0-9]*"
                  value={bad}
                  onChange={(e) => onNumberChange(e, setBad)}
                  placeholder="e.g., 3"
                  className="w-full rounded-xl bg-slate-900 border border-white/10 px-4 py-3 outline-none focus:ring-2 focus:ring-indigo-400/60 focus:border-indigo-400/60 font-medium"
                />
              </div>

              <div className="flex gap-3 pt-2">
                <button onClick={loadExample} className="flex-1 rounded-xl bg-indigo-500 hover:bg-indigo-400 active:bg-indigo-600 transition px-4 py-3 font-medium">Load example</button>
                <button onClick={reset} className="flex-1 rounded-xl bg-slate-700 hover:bg-slate-600 active:bg-slate-800 transition px-4 py-3 font-medium">Reset</button>
              </div>

              <p className="text-xs text-slate-400 pt-1">Only non‑negative integers are accepted.</p>
            </div>
          </section>

          <section className="rounded-2xl bg-slate-950/40 backdrop-blur border border-white/10 p-5 shadow-xl shadow-black/30">
            <h2 className="text-lg font-medium mb-4">Result</h2>
            <div className="space-y-3">
              <div className="grid grid-cols-2 gap-2 text-sm">
                <span className="text-slate-300">Total feedback (Good + Bad)</span>
                <span className="text-right font-semibold">{total}</span>
              </div>
              <div className="grid grid-cols-2 gap-2 text-sm">
                <span className="text-slate-300">Fraction (Good ÷ Total)</span>
                <span className="text-right font-mono">{total > 0 ? ratio.toPrecision(6) : "—"}</span>
              </div>
              <div className="border-t border-white/10 my-3" />
              <div className="flex items-end justify-between">
                <div>
                  <p className="text-slate-300 text-sm mb-1">CSAT percentage</p>
                  <p className="text-xs text-slate-400">(Good ÷ (Good + Bad)) × 100</p>
                </div>
                <div className="text-right">
                  <div className="text-4xl md:text-5xl font-extrabold tracking-tight">{fmt(percent, 2)}%</div>
                  <div className="text-xs text-slate-400">{Number.isFinite(percent) && total > 0 ? `${fmt(ratio * 100, 4)}% raw` : ""}</div>
                </div>
              </div>

              <progress
                className="w-full h-3 rounded-full overflow-hidden [&::-webkit-progress-bar]:bg-slate-800 [&::-webkit-progress-value]:bg-indigo-500"
                value={Number.isFinite(percent) ? Math.max(0, Math.min(100, percent)) : 0}
                max={100}
                aria-label="CSAT percentage"
              />

              <div className="text-xs text-slate-400">
                {total > 0 ? (
                  <>
                    Example math: <span className="font-mono">{parsedGood}</span> ÷ <span className="font-mono">{total}</span> = <span className="font-mono">{ratio.toFixed(16)}</span> → × 100 = <span className="font-mono">{fmt(percent, 2)}%</span>
                  </>
                ) : (
                  <span>Enter values to see the calculation.</span>
                )}
              </div>
            </div>
          </section>
        </div>

        <footer className="mt-6 text-center text-xs text-slate-400">
          Tip: CSAT is sensitive to sample size; small totals can swing the % a lot.
        </footer>
      </div>
    </div>
  );
}
