import sodium from "libsodium-wrappers";
import { ApplicationSettings } from "@nativescript/core";
export type EncryptedBlob = { n: string; c: string; a?: string };
export class CryptoService {
  private ready = false;
  private masterKey: Uint8Array | null = null;
  private static KDF_RECORD_KEY = "kdf_params_v1";
  private static WRAPPED_MK_KEY = "wrapped_master_key_v1";
  async init() { if (this.ready) return; await sodium.ready; this.ready = true; }
  private deriveKey(passphrase: string, salt: Uint8Array, ops?: number, mem?: number) {
    const opslimit = ops ?? sodium.crypto_pwhash_OPSLIMIT_MODERATE;
    const memlimit = mem ?? sodium.crypto_pwhash_MEMLIMIT_MODERATE;
    return sodium.crypto_pwhash(32, passphrase, salt, opslimit, memlimit, sodium.crypto_pwhash_ALG_DEFAULT);
  }
  async unlockWithPassphrase(passphrase: string): Promise<boolean> {
    await this.init();
    const kdfMetaStr = ApplicationSettings.getString(CryptoService.KDF_RECORD_KEY);
    const wrappedStr = ApplicationSettings.getString(CryptoService.WRAPPED_MK_KEY);
    if (!kdfMetaStr || !wrappedStr) {
      const salt = sodium.randombytes_buf(16);
      const mk = sodium.randombytes_buf(32);
      const k = this.deriveKey(passphrase, salt);
      const nonce = sodium.randombytes_buf(sodium.crypto_aead_xchacha20poly1305_ietf_NPUBBYTES);
      const aad = sodium.from_string("ns-secure-notes-v1");
      const ct = sodium.crypto_aead_xchacha20poly1305_ietf_encrypt(mk, aad, null, nonce, k);
      ApplicationSettings.setString(CryptoService.KDF_RECORD_KEY, JSON.stringify({ s: sodium.to_base64(salt), o: sodium.crypto_pwhash_OPSLIMIT_MODERATE, m: sodium.crypto_pwhash_MEMLIMIT_MODERATE }));
      ApplicationSettings.setString(CryptoService.WRAPPED_MK_KEY, JSON.stringify({ n: sodium.to_base64(nonce), c: sodium.to_base64(ct), a: sodium.to_base64(aad) }));
      this.masterKey = mk; return true;
    }
    try {
      const { s, o, m } = JSON.parse(kdfMetaStr);
      const salt = sodium.from_base64(s);
      const k = this.deriveKey(passphrase, salt, o, m);
      const { n, c, a } = JSON.parse(wrappedStr) as EncryptedBlob;
      const nonce = sodium.from_base64(n);
      const ct = sodium.from_base64(c);
      const aad = a ? sodium.from_base64(a) : sodium.from_string("ns-secure-notes-v1");
      const mk = sodium.crypto_aead_xchacha20poly1305_ietf_decrypt(null, ct, aad, nonce, k);
      this.masterKey = mk; return true;
    } catch (_) { this.masterKey = null; return false; }
  }
  lock() { if (this.masterKey) this.masterKey.fill(0); this.masterKey = null; }
  async changePassphrase(newPass: string) {
    if (!this.masterKey) throw new Error("Not unlocked");
    const salt = sodium.randombytes_buf(16);
    const k = this.deriveKey(newPass, salt);
    const nonce = sodium.randombytes_buf(sodium.crypto_aead_xchacha20poly1305_ietf_NPUBBYTES);
    const aad = sodium.from_string("ns-secure-notes-v1");
    const ct = sodium.crypto_aead_xchacha20poly1305_ietf_encrypt(this.masterKey, aad, null, nonce, k);
    ApplicationSettings.setString(CryptoService.KDF_RECORD_KEY, JSON.stringify({ s: sodium.to_base64(salt), o: sodium.crypto_pwhash_OPSLIMIT_MODERATE, m: sodium.crypto_pwhash_MEMLIMIT_MODERATE }));
    ApplicationSettings.setString(CryptoService.WRAPPED_MK_KEY, JSON.stringify({ n: sodium.to_base64(nonce), c: sodium.to_base64(ct), a: sodium.to_base64(aad) }));
  }
  encryptJSON(obj: any): EncryptedBlob {
    if (!this.masterKey) throw new Error("Locked");
    const nonce = sodium.randombytes_buf(sodium.crypto_aead_xchacha20poly1305_ietf_NPUBBYTES);
    const aad = sodium.from_string("note:v1");
    const pt = sodium.from_string(JSON.stringify(obj));
    const ct = sodium.crypto_aead_xchacha20poly1305_ietf_encrypt(pt, aad, null, nonce, this.masterKey);
    return { n: sodium.to_base64(nonce), c: sodium.to_base64(ct), a: sodium.to_base64(aad) };
  }
  decryptJSON(blob: EncryptedBlob): any {
    if (!this.masterKey) throw new Error("Locked");
    const nonce = sodium.from_base64(blob.n);
    const ct = sodium.from_base64(blob.c);
    const aad = blob.a ? sodium.from_base64(blob.a) : sodium.from_string("note:v1");
    const pt = sodium.crypto_aead_xchacha20poly1305_ietf_decrypt(null, ct, aad, nonce, this.masterKey);
    return JSON.parse(sodium.to_string(pt));
  }
}
export const cryptoService = new CryptoService();