import { knownFolders, path } from "@nativescript/core";
const Sqlite = require("nativescript-sqlite");
export type NoteRow = { id: number; updated_at: number; blob: string };
export class DB {
  private db: any;
  async init() {
    const docs = knownFolders.documents();
    const dbPath = path.join(docs.path, "secure_notes.db");
    this.db = await new Sqlite(dbPath);
    await this.db.execute(`CREATE TABLE IF NOT EXISTS notes (id INTEGER PRIMARY KEY AUTOINCREMENT, updated_at INTEGER NOT NULL, blob TEXT NOT NULL);`);
    await this.db.execute("CREATE INDEX IF NOT EXISTS idx_notes_updated ON notes(updated_at DESC);");
  }
  async list(): Promise<NoteRow[]> {
    const rows = await this.db.all("SELECT id, updated_at, blob FROM notes ORDER BY updated_at DESC");
    return rows.map((r) => ({ id: r[0], updated_at: r[1], blob: r[2] }));
  }
  async get(id: number): Promise<NoteRow | null> {
    const rows = await this.db.all("SELECT id, updated_at, blob FROM notes WHERE id = ?", [id]);
    if (!rows || !rows.length) return null;
    const r = rows[0];
    return { id: r[0], updated_at: r[1], blob: r[2] };
  }
  async upsert(id: number | null, blob: string): Promise<number> {
    const ts = Date.now();
    if (id) { await this.db.execute("UPDATE notes SET updated_at = ?, blob = ? WHERE id = ?", [ts, blob, id]); return id; }
    await this.db.execute("INSERT INTO notes (updated_at, blob) VALUES (?, ?)", [ts, blob]);
    const row = await this.db.get("SELECT last_insert_rowid() as id");
    return row[0];
  }
  async remove(id: number) { await this.db.execute("DELETE FROM notes WHERE id = ?", [id]); }
}
export const db = new DB();