// session-transportd (v2): Production Bun/Elysia daemon using Session.js
// Changes vs v1: pinned package versions and `await ready` fix.
import { Session, Poller, ready } from "@session.js/client";
import { FileKeyvalStorage } from "@session.js/file-keyval-storage";
import { generateSeedHex } from "@session.js/keypair";
import { encode as encodeMnemonic } from "@session.js/mnemonic";
import { BunNetwork } from "@session.js/bun-network";
import { Elysia } from "elysia";

type Json = Record<string, unknown> | Array<unknown> | string | number | boolean | null;

const HOST = process.env.HOST ?? "127.0.0.1";
const PORT = Number(process.env.PORT ?? 8787);
const STORAGE_DB = process.env.STORAGE_DB ?? "./session-storage.db";
const MNEMONIC_FILE = process.env.MNEMONIC_FILE ?? "./mnemonic.txt";
const DISPLAY_NAME = process.env.DISPLAY_NAME ?? "session-transportd";
const API_TOKEN = process.env.API_TOKEN ?? "";
const PROXY_URL = process.env.PROXY_URL ?? "";

const log = (level: "info" | "warn" | "error", message: string, meta: Record<string, Json> = {}) => {
  const out = { ts: new Date().toISOString(), level, msg: message, ...meta };
  console.log(JSON.stringify(out));
};

class LruSet {
  private max: number;
  private map = new Map<string, number>();
  constructor(max = 10000) { this.max = max; }
  has(k: string) { return this.map.has(k); }
  add(k: string) {
    if (this.map.has(k)) this.map.delete(k);
    this.map.set(k, Date.now());
    if (this.map.size > this.max) {
      const oldest = this.map.keys().next().value;
      this.map.delete(oldest);
    }
  }
}

async function loadMnemonic(): Promise<string> {
  if (process.env.MNEMONIC && process.env.MNEMONIC.trim().length) {
    return process.env.MNEMONIC.trim();
  }
  try {
    const text = await Bun.file(MNEMONIC_FILE).text();
    if (text.trim().length) return text.trim();
  } catch {}
  const seedHex = generateSeedHex();
  const mnemonic = encodeMnemonic(seedHex);
  await Bun.write(MNEMONIC_FILE, mnemonic);
  log("info", "Generated new mnemonic and saved to file", { MNEMONIC_FILE });
  return mnemonic;
}

function bearerOk(req: Request): boolean {
  if (!API_TOKEN) return true;
  const h = req.headers.get("authorization") || req.headers.get("Authorization");
  if (!h) return false;
  const [scheme, token] = h.split(/\s+/, 2);
  return scheme?.toLowerCase() === "bearer" && token === API_TOKEN;
}

function wsTokenOk(url: string): boolean {
  if (!API_TOKEN) return true;
  try {
    const u = new URL(url, `http://${HOST}:${PORT}`);
    return u.searchParams.get("token") === API_TOKEN;
  } catch { return false; }
}

await ready; // initialize crypto libs (IMPORTANT)

const session = new Session({
  storage: new FileKeyvalStorage({ filePath: STORAGE_DB }),
  network: PROXY_URL ? new BunNetwork({ proxy: PROXY_URL }) : new BunNetwork()
});

const mnemonic = await loadMnemonic();
session.setMnemonic(mnemonic, DISPLAY_NAME);
const sessionId = session.getSessionID();

const poller = new Poller(); // default ~2.5s interval
session.addPoller(poller);

type OutMsg = { id: string; from: string; text?: string; timestamp: number };
const inbox: OutMsg[] = [];
const seen = new LruSet(10000);
const sockets = new Set<WebSocket>();

session.on("message", (msg: any) => {
  const id: string = (msg.id as string) || `${msg.timestamp}:${msg.from}`;
  if (seen.has(id)) return;
  seen.add(id);
  const out: OutMsg = { id, from: msg.from, text: msg.text, timestamp: msg.timestamp };
  inbox.push(out);
  const payload = JSON.stringify({ type: "message", ...out });
  for (const ws of sockets) { try { ws.send(payload); } catch {} }
});

session.on("messageRequestApproved", (evt: any) => {
  const payload = JSON.stringify({ type: "messageRequestApproved", ...evt });
  for (const ws of sockets) { try { ws.send(payload); } catch {} }
});

const app = new Elysia()
  .derive(({ request, set }) => {
    if (!bearerOk(request)) { set.status = 401; return { authed: false }; }
    return { authed: true };
  })
  .get("/v1/id", () => ({ id: sessionId }))
  .get("/v1/healthz", () => ({ ok: true }))
  .post("/v1/send", async ({ body, set }) => {
    try {
      const { to, text, data, replyTo } = body as { to?: string, text?: string, data?: any, replyTo?: number };
      if (!to || (!text && typeof data === "undefined")) { set.status = 400; return { error: "to and text|data required" }; }
      const payload = typeof text === "string" ? text : JSON.stringify(data);
      const res = await session.sendMessage({
        to, text: payload,
        ...(replyTo ? { replyToMessage: { timestamp: replyTo, author: sessionId, text: "" } } : {})
      });
      log("info", "sent message", { to, timestamp: res.timestamp });
      return res;
    } catch (e: any) { set.status = 500; log("error", "send failed", { error: String(e) }); return { error: "send failed", detail: String(e) }; }
  })
  .post("/v1/accept", async ({ body, set }) => {
    try {
      const { from } = body as { from?: string };
      if (!from) { set.status = 400; return { error: "from required" }; }
      // @ts-ignore - available on Session.js
      await session.acceptConversationRequest({ from });
      log("info", "accepted conversation request", { from });
      return { ok: true };
    } catch (e: any) { set.status = 500; log("error", "accept failed", { error: String(e) }); return { error: "accept failed", detail: String(e) }; }
  })
  .get("/v1/recv", ({ query }) => {
    const n = Math.max(1, Math.min(Number(query.n ?? 100), 1000));
    const batch = inbox.splice(0, n);
    return { messages: batch };
  })
  .ws("/v1/ws", {
    open(ws) {
      if (!wsTokenOk(ws.data.request.url)) { try { ws.close(1008, "Unauthorized"); } catch {}; return; }
      sockets.add(ws);
      ws.send(JSON.stringify({ type: "hello", id: sessionId }));
    },
    async message(ws, data) {
      if (!data) return;
      try {
        const msg = JSON.parse(String(data));
        if (msg?.type === "send" && msg.to && (msg.text || typeof msg.data !== "undefined")) {
          const payload = typeof msg.text === "string" ? msg.text : JSON.stringify(msg.data);
          const res = await session.sendMessage({ to: msg.to, text: payload });
          ws.send(JSON.stringify({ type: "sent", ...res }));
        }
      } catch {}
    },
    close(ws) { sockets.delete(ws); }
  })
  .listen({ hostname: HOST, port: PORT });

log("info", "session-transportd started", { HOST, PORT, sessionId });

const shutdown = async (signal: string) => {
  log("warn", "shutting down", { signal });
  try { poller.stopPolling(); } catch {}
  try { for (const ws of sockets) ws.close(); } catch {}
  try { await app.stop(); } catch {}
  process.exit(0);
};

process.on("SIGINT", () => shutdown("SIGINT"));
process.on("SIGTERM", () => shutdown("SIGTERM"));
